/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;
import java.util.List;

import com.inet.report.Area;
import com.inet.report.Chart2;
import com.inet.report.DatabaseField;
import com.inet.report.Engine;
import com.inet.report.Fields;
import com.inet.report.Section;
import com.inet.report.SummaryField;
import com.inet.report.chart.dataset.OneGroupDataset;
import com.inet.report.chart.plot.BarPlot;
import com.inet.report.chart.plot.BarStyle;

/**
 * This sample explains how to add a chart to report header section of an existing report file. The chart displays a sum
 * of "Last Year's Sales" per customer.<br>
 * <br>
 * If you want to modify an existing chart in a report, you may only get the chart element from elementsV Vector. You
 * can get this Vector with method getElementsV() in class Section. Then you go through the {@link List} with a loop and hold
 * on every instance of Chart and look for whether it is the right one or not (i.e. by name of the Chart). Then you can
 * store it with a cast in a Chart reference and work with it.<br>
 * <br>
 * You may change the location of the report file that is used in this sample. The used report file is "mainrep.rpt" and
 * is included with this RDC sample suite.
 * @see Chart2
 */
public class ChartSample extends RDCSample {

    // Change this before running the sample, if the URL does not match 
    // the sample report "mainrep.rpt" that comes with this RDC sample suite. 
    private static final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Add a chart to report header section of an existing report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            // Create a new Engine
            Engine eng = new Engine( exportFmt );
            // Set the name of the report file
            //eng.setReportFile("file:" + System.getProperty("user.dir") + rptFileName);
            URL reportURL = getClass().getResource( '/' + MAINREPORT );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( MAINREPORT );
            }

            // Get area where the chart should placed -> report header
            Area area = eng.getArea( "RH" );
            // Get first section of report header area
            Section section = area.getSection( 0 );
            // In the rpt file the first report header section is suppressed,
            // but we will add the chart to it, therefore we make it visible.
            section.setSuppress( false );

            // The report mainrep.rpt contains too much data for this little chart,
            // therefore we set a record selection formula to reduce the number of rows
            eng.setSF( "{Customers.CustomerID} in 12311 to 12315" );

            // Get all fields defined in this report
            Fields dbFields = eng.getFields();

            // Get two database fields for chart
            DatabaseField dbField1 = dbFields.getDatabaseField( "Customers.ContactName" );
            DatabaseField dbField2 = dbFields.getDatabaseField( "Customers.LastYearsSales" );

            // Add chart and sets the type of chart (3D bar chart)
            Chart2 chart = section.addChart2( BarStyle.BAR3D, 1000, 100, 8000, 4000 );

            // Add a source to x axis
            BarPlot plot = (BarPlot)chart.getPlot();

            OneGroupDataset dataset = new OneGroupDataset( chart );
            dataset.setCategoryGroup( dbField1 );
            dataset.addDataField( SummaryField.SUM, dbField2, null, 0 );
            plot.setDataset( dataset );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new ChartSample().initUI();
    }
}
